# A tool for visualizing YOLO format object detection annotations on images with customizable class colors and confidence thresholds.

import os
import cv2

# Class names and colors
class_names = {
    0: {"name": "White-eared Kob", "color": (0, 0, 255)},
    1: {"name": "Mongalla Gazelle", "color": (0, 255, 0)},
    2: {"name": "Tiang", "color": (0, 255, 255)},
    3: {"name": "Reedbuck", "color": (255, 0, 255)},
    4: {"name": "Others", "color": (255, 0, 0)},
}

# Whether to display class names
show_class_name = False

# Whether to use confidence score (True: YOLOv8 format with conf, False: YOLOv5 etc. without conf)
use_conf = True

# Visualization function
def visualize_yolo(image_folder, label_folder, output_folder, conf_threshold=0.0):
    os.makedirs(output_folder, exist_ok=True)

    for file in os.listdir(image_folder):
        if not file.lower().endswith((".jpg", ".png", ".bmp")):
            continue

        image_path = os.path.join(image_folder, file)
        label_path = os.path.join(label_folder, os.path.splitext(file)[0] + ".txt")
        output_path = os.path.join(output_folder, file)

        image = cv2.imread(image_path)
        if image is None:
            print(f"Failed to read image: {image_path}")
            continue

        h, w = image.shape[:2]

        if not os.path.exists(label_path):
            print(f"Label file not found: {label_path}")
            continue

        with open(label_path, 'r') as f:
            lines = f.readlines()

        for line in lines:
            parts = line.strip().split()
            if (use_conf and len(parts) != 6) or (not use_conf and len(parts) != 5):
                print(f"Invalid label format: {label_path} -> {line}")
                continue

            class_id = int(parts[0])
            if class_id not in class_names:
                continue

            x, y, bw, bh = map(float, parts[1:5])
            conf = float(parts[5]) if use_conf else 1.0
            if conf < conf_threshold:
                continue

            # Coordinate conversion
            x1 = int((x - bw / 2) * w)
            y1 = int((y - bh / 2) * h)
            x2 = int((x + bw / 2) * w)
            y2 = int((y + bh / 2) * h)

            color = class_names[class_id]["color"]
            cv2.rectangle(image, (x1, y1), (x2, y2), color, 20)

            if show_class_name:
                name = class_names[class_id]["name"]
                cv2.putText(image, name, (x1, y1 - 5), cv2.FONT_HERSHEY_SIMPLEX, 0.8, color, 20)

        cv2.imwrite(output_path, image)
        print(f"Saved image: {output_path}")

# ==== Set paths and run ====
image_folder = r'D:/***'           # Original image path
label_folder = r'D:/***'           # YOLO label path (.txt)
output_folder = r'D:/***'  

visualize_yolo(image_folder, label_folder, output_folder, conf_threshold=0.0)
